// (c) 1999 - 2025 OneSpan North America Inc. All rights reserved.


/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


import UIKit
import MSSOrchestration
import MSSNotificationClient

class MainViewController: UITableViewController {
    private enum CellIndex: Int {
        case selectUser = 0,
             localAuthentication,
             localTransaction,
             changePassword,
             getInformation,
             CDDCMessage
    }
    
    private let numberOfCells = 6
    private let estimatedRowHeight: CGFloat = 600
    private let orchestrationDelegate = OrchestrationSampleDelegate()
    private let serverCallHandler = ServerCallHandler()
    private let notificationManager = NotificationManager()
    
    private var orchestrator: Orchestrator?
    private var progressDialog: UIView?
    
    @IBOutlet private var content: UITableView!
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        print("Version: \(OrchestratorBuilder.version)")
        
        setupObservers()
        setupOrchestration()
        
        // Clean user default stored values
        // After a backup/resore a desynchronization can occured
        // between data stored in the library and in the Sample
        let isUserListEmpty: Bool
        do {
            isUserListEmpty = try orchestrator?.userManager.users().count == 0
        } catch {
            print(error)
            isUserListEmpty = false;
        }
        if SharedPreferenceStorage.getActivatedUser() != nil && isUserListEmpty {
            SharedPreferenceStorage.clean()
        }
        
        setupView()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        content.reloadData()
        updateNotificationID()
    }
    
    // MARK: Setup
    private func setupView() {
        content.delegate = self
        content.dataSource = self
        content.alwaysBounceHorizontal = false
        content.rowHeight = UITableView.automaticDimension
        content.estimatedRowHeight = estimatedRowHeight
        title = "title_activity_main".localized
        navigationController?.navigationBar.titleTextAttributes = [.foregroundColor: UIColor.white]
    }
    
    private func setupOrchestration() {
        orchestrationDelegate.progressDialog = progressDialog
        orchestrationDelegate.viewController = self
        orchestrator = OrchestratorUtils.getOrchestrator(delegate: orchestrationDelegate)
        serverCallHandler.viewController = self
        serverCallHandler.orchestrator = orchestrator
    }
    
    private func setupObservers() {
        NotificationCenter.default.addObserver(self, selector: #selector(newNotificationId), name: .managerNewDeviceToken, object: nil)
        NotificationCenter.default.addObserver(self, selector: #selector(newNotificationReceived), name: .managerPending, object: nil)
        NotificationCenter.default.addObserver(self, selector: #selector(onNotificationError), name: .managerErrorWithParse, object: nil)
    }
    
    // MARK: Navigation
    override func prepare(for segue: UIStoryboardSegue, sender: Any?) {
        if let sender = sender as? OneSpanNotification,
           let destination = segue.destination as? RemoteViewController,
           segue.identifier == Constants.Segues.goToRemoteSegue {
            destination.setCommand(sender.oneSpanContent)
        }
        
        if let sender = sender as? NSNumber,
           let destination = segue.destination as? RegistrationViewController,
           segue.identifier == Constants.Segues.goToRegistrationSegue {
            destination.hidesBackButton = sender.boolValue
        }
    }
    
    private func updateUserList(_ cell: SelectTableViewCell, _ list: [OrchestrationUser], _ user: String) {
        let index = list.firstIndex { $0.identifier == user } ?? 0
        cell.setListForSelection(list, index)
    }
    
    private func actionOnDeleteAllButton() {
        let confirmButtonAction = {
            do {
                try self.orchestrator?.userManager.deleteAllUsers()
                SharedPreferenceStorage.clean()
                SharedPreferenceStorage.storeActivatedUser(activatedUser: nil)
            } catch {
                print(error)
            }
            self.tableView.reloadData()
        }
        
        UIUtils.displayAlert(controller: self,
                             title: "confirm_delete_all_title".localized,
                             message: "confirm_delete_all_message".localized,
                             firstButton: "confirm_delete_all_yes".localized,
                             firstAction: confirmButtonAction,
                             secondButton: "confirm_delete_all_no".localized)
    }
    
    private func actionOnDeleteButton(cell: SelectTableViewCell) {
        guard let currentUser = SharedPreferenceStorage.getActivatedUser() else {
            assertionFailure("Can't get the current activated user.")
            return
        }
        
        let message = String(format: "confirm_delete_user_message".localized, currentUser)
        
        let confirmButtonAction = {
            do {
                try self.orchestrator?.userManager.delete(user: OrchestrationUser(identifier: currentUser, domain: nil))
                SharedPreferenceStorage.removeNotificationId(user: currentUser)
                
                if let userList = try self.orchestrator?.userManager.users(), let activatedUser = userList.first?.identifier {
                    SharedPreferenceStorage.storeActivatedUser(activatedUser: activatedUser)
                    self.updateUserList(cell, userList, activatedUser)
                } else {
                    SharedPreferenceStorage.storeActivatedUser(activatedUser: nil)
                }
            } catch {
                print(error)
            }
            
            self.tableView.reloadData()
        }
        
        UIUtils.displayAlert(controller: self,
                             title: "confirm_delete_user_title".localized,
                             message: message,
                             firstButton: "confirm_delete_user_yes".localized,
                             firstAction: confirmButtonAction,
                             secondButton: "confirm_delete_user_no".localized)
    }
    
    // MARK: Private notification id
    @objc private func newNotificationId(notification: Notification) {
        var notificationId = ""
        
        do {
            // Get activated user and previously saved notification ID
            guard let activatedUser = SharedPreferenceStorage.getActivatedUser(), let deviceToken = notification.object as? Data else { return }
            
            notificationId = try NotificationClientSDK.oneSpanNotificationIdentifier(for: deviceToken)
            let previousNotificationId = SharedPreferenceStorage.storedNotificationId(user: activatedUser)
            
            if previousNotificationId != nil && previousNotificationId == notificationId {
                return
            }
            
            // Send
            SharedPreferenceStorage.storeLastRetrievedNotificationId(notificationId)
        } catch let error as NotificationError {
            let message = "\("notification_identifier_error".localized): \(error.formatted)"
            NotificationCenter.default.post(name: .managerErrorWithParse, object: message)
            print(message)
        } catch {
            assertionFailure("notification_error_not_known".localized)
        }
        
        newNotificationID(identifier: notificationId)
    }
    
    private func newNotificationID(identifier: String) {
        // Get activated user from shared preferences
        guard let activatedUser = SharedPreferenceStorage.getActivatedUser() else { return }
        
        // Prepare params object
        let orchestrationUser = OrchestrationUser(identifier: activatedUser, domain: nil)
        let params = NotificationRegistrationParameters(user: orchestrationUser, notificationIdentifier: identifier, delegate: self)
        
        // start notification registration process
        displayProgress(message: "dialog_progress_notification".localized)
        orchestrator?.startNotificationRegistration(with: params)
    }
    
    // MARK: Private notification received
    @objc private func newNotificationReceived(notification: Notification) {
        // Open login Screen
        performSegue(withIdentifier: Constants.Segues.goToRemoteSegue, sender: notification.object)
    }
    
    @objc private func onNotificationError(notification: Notification) {
        let message = notification.object as? String ?? "dialog_error_content_notification".localized
        UIUtils.hideProgress(progressDialog)
        displayAlert(title: "dialog_error_title".localized, message: message)
        print(message)
    }
    
    // MARK: Private utils
    private func updateNotificationID() {
        // Handle possible incoming notification
        guard let lastRetrievedStoredNotificationId = SharedPreferenceStorage.getLastRetrievedStoredNotificationId() else {
            notificationManager.registerForNotifications()
            return
        }
        
        // Check if the notification id have changed
        guard let user = SharedPreferenceStorage.getActivatedUser() else { return }
        
        let storedNotificationId = SharedPreferenceStorage.storedNotificationId(user: user)
        if lastRetrievedStoredNotificationId != storedNotificationId {
            newNotificationID(identifier: lastRetrievedStoredNotificationId)
        }
    }
    
    private func displayAlert(title: String, message: String) {
        var controller: UIViewController? = self
        while controller?.presentingViewController != nil {
            controller = controller?.presentingViewController
        }
        
        guard let viewController = controller else { return }
        UIUtils.displayAlert(controller: viewController, title: title, message: message)
    }
    
    private func displayProgress(message: String) {
        if progressDialog != nil {
            UIUtils.hideProgress(progressDialog)
        }
        
        progressDialog = UIUtils.displayProgress(controller: self, message: message)
        orchestrationDelegate.progressDialog = progressDialog
    }
    
    private func sendCommandToServer(_ command: String) {
        serverCallHandler.progressDialog = progressDialog
        serverCallHandler.sendCommandToServer(command)
    }
    
    deinit {
        NotificationCenter.default.removeObserver(self)
    }
}

// tableview delegate
extension MainViewController {
    override func tableView(_ tableView: UITableView, heightForRowAt indexPath: IndexPath) -> CGFloat {
        return SharedPreferenceStorage.getActivatedUser() != nil ? UITableView.automaticDimension: tableView.contentSize.height
    }
}

// tableview datasource
extension MainViewController {
    override func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return SharedPreferenceStorage.getActivatedUser() == nil ? 1 : numberOfCells
    }
    
    override func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        let userActivity = SharedPreferenceStorage.getActivatedUser()
        var identifier: String
        
        if SharedPreferenceStorage.getActivatedUser() == nil {
            identifier = Constants.CellIdentifiers.registerCell
        } else if indexPath.row == 0 {
            identifier = Constants.CellIdentifiers.selectUser
        } else {
            identifier = Constants.CellIdentifiers.actionCell
        }
        
        guard let cell = tableView.dequeueReusableCell(withIdentifier: identifier) as? TableViewCell else {
            return UITableViewCell()
        }
        
        weak var welf: MainViewController? = self
        
        if userActivity == nil {
            (cell as? RegisterTableViewCell)?.configure {
                welf?.performSegue(withIdentifier: Constants.Segues.goToRegistrationSegue, sender: true)
            }
        } else {
            guard let userActivity = userActivity else {
                return UITableViewCell()
            }
            
            configure(cell: cell, in: indexPath.row, userActivity: userActivity)
        }
        
        cell.setCellIndex(indexPath.row)
        cell.setNeedsUpdateConstraints()
        cell.updateConstraintsIfNeeded()
        
        return cell
    }
    
    private func configure(cell: TableViewCell, in row: Int, userActivity: String) {
        weak var welf: MainViewController? = self
        
        switch row {
        case CellIndex.selectUser.rawValue:
            if let sCell = cell as? SelectTableViewCell {
                let list: [OrchestrationUser]
                do {
                    list = try orchestrator?.userManager.users() ?? []
                } catch {
                    list = []
                    print(error)
                }
                
                sCell.setTitle("su_title".localized)
                sCell.setButtonTitle("su_button".localized)
                sCell.setLeftButtonTitle("su_left_button".localized)
                sCell.setBottomButtonTitle("su_bottom_button".localized)
                sCell.setTitleForSelection("su_selection_title".localized)
                sCell.setMessageForSelection("su_selection_message".localized)
                sCell.setPresenter(self)
                
                updateUserList(sCell, list, userActivity)
                
                sCell.setButtonAction {
                    welf?.performSegue(withIdentifier: Constants.Segues.goToRegistrationSegue, sender: NSNumber(value: false))
                }
                
                sCell.setLeftButtonAction {
                    welf?.actionOnDeleteButton(cell: sCell)
                }
                
                sCell.setBottomButtonAction {
                    welf?.actionOnDeleteAllButton()
                }
                
                sCell.setActionOnSelection { (item: SelectableItem) in
                    if let user = item as? OrchestrationUser {
                        SharedPreferenceStorage.storeActivatedUser(activatedUser: user.identifier)
                        welf?.updateNotificationID()
                    }
                }
            }
            
        case CellIndex.localAuthentication.rawValue:
            (cell as? ActionTableViewCell)?.configure(
                title: "la_title".localized,
                subtitle: "la_subtitle".localized,
                buttonTitle: "la_button".localized,
                buttonAction: {
                    welf?.performSegue(withIdentifier: Constants.Segues.goToLocalAuthSegue, sender: nil)
                }
            )
            
        case CellIndex.localTransaction.rawValue:
            (cell as? ActionTableViewCell)?.configure(
                title: "lt_title".localized,
                subtitle: "lt_subtitle".localized,
                buttonTitle: "lt_button".localized,
                buttonAction: {
                    welf?.performSegue(withIdentifier: Constants.Segues.goToLocalTransactionSegue, sender: nil)
                }
            )
            
        case CellIndex.changePassword.rawValue:
            (cell as? ActionTableViewCell)?.configure(
                title: "cp_title".localized,
                subtitle: "cp_subtitle".localized,
                buttonTitle: "cp_button".localized,
                buttonAction: {
                    welf?.performSegue(withIdentifier: Constants.Segues.goToChangePasswordSegue, sender: nil)
                }
            )
            
        case CellIndex.getInformation.rawValue:
            (cell as? ActionTableViewCell)?.configure(
                title: "gi_title".localized,
                subtitle: "gi_subtitle".localized,
                buttonTitle: "gi_button".localized,
                buttonAction: {
                    welf?.performSegue(withIdentifier: Constants.Segues.goToGetInfoSegue, sender: nil)
                }
            )
        case CellIndex.CDDCMessage.rawValue:
            (cell as? ActionTableViewCell)?.configure(
                title: "cddc_title".localized,
                subtitle: "cddc_subtitle".localized,
                buttonTitle: "cddc_button".localized,
                buttonAction: {
                    welf?.performSegue(withIdentifier: Constants.Segues.gotoCDDCMessage, sender: nil)
                }
            )
            
        default:
            break
        }
    }
}

extension MainViewController: NotificationRegistrationDelegate {
    func orchestrator(_ orchestrator: Orchestrator, didCompleteNotificationRegistrationStepWith command: String) {
        print("onNotificationRegistrationStepCompleteWithCommand:", command)
        sendCommandToServer(command)
    }
    
    func orchestrator(_ orchestrator: Orchestrator, didFinishNotificationRegistrationWithSuccessFor user: OrchestrationUser, notificationIdentifier: String) {
        // Store notification ID in shared preferences
        SharedPreferenceStorage.storeNotificationId(notificationID: notificationIdentifier, user: user.identifier)
        
        // Hide progress dialog and show success alert
        UIUtils.hideProgress(progressDialog)
        
        displayAlert(title: "dialog_title_notification".localized, message: "dialog_content_notification".localized)
    }
}
